/*  
 * Copyright (C) 2005 Martin Pischky (mailto:martin@pischky.de)  
 *
 * This file (HexFileReader.java) is part of IntelHexFile.
 *
 * IntelHexFile is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 */

/*
 * $Log: HexFileReader.java,v $
 * Revision 1.1  2005/03/27 12:29:58  pischky
 * added
 *
 */

package org.fremo.ihx;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.Reader;

/**
 * @author martin
 */
public class HexFileReader {

    private Reader rdr;
    
    /**
     * Construct a new HexFileReader from a given Reader.
     * The Reader should be opened.
     */
    public HexFileReader(Reader rdr) {
        super();
        this.setRdr(rdr);
    }

    /**
     * 
     */
    public HexFileReader() {
        super();
    }

    /**
     * close should be called.
     * @throws IOException
     */
    public void close() 
    throws IOException 
    {
        this.getRdr().close();
    }
    
    /**
     * Read the file and create a Memory object of the data contained therein.
     * @return the memory object representing this file
     * @throws IOException
     * @throws InvalidHexFileException
     */
    public Memory read() 
    throws IOException, InvalidHexFileException
    {
        Memory mem;
        BufferedReader r;
        mem = new ChunkedMemory();
        if( getRdr() instanceof BufferedReader ) {
            r = (BufferedReader) getRdr();
        } else {
            r = new BufferedReader(getRdr());
        }
        HexFileReader.readIntelHexFile(mem, r);
        if( ! (getRdr() instanceof BufferedReader) ) {
            r.close();
        }
        return mem;
    }
    
    /**
     * @param mem
     * @param rdr
     * @throws IOException
     */
    public static void readIntelHexFile( Memory mem, BufferedReader rdr ) 
    throws IOException, InvalidHexFileException
    {
        boolean read;
        do {
            String line = rdr.readLine();
            if( line == null ) {
                throw new InvalidHexFileException( "unexpected end of file" );
            }
            read = parseLine( mem, line );
        } while( read );
    }

    /**
     * returns false on EOF.
     * @param mem
     * @param line
     * @return false: valid eof in hex file dound
     * @throws InvalidHexFileException
     */
    private static boolean parseLine( Memory mem, String line ) 
    throws InvalidHexFileException 
    {
        HexFileLine hl = parseHexFileLine( line );
        if( hl.getRecordType() == HexFileLine.EOF_RECORD ) {
            // do not check address field (may be a load address)
            return false;
        }
        for (int i = 0; i < hl.getRecordLength(); i++) {
            mem.set( hl.getAddress()+i, hl.getData(i) );
        }
        return true;
    }
    
    private static HexFileLine parseHexFileLine(String line) 
    throws InvalidHexFileException
    {
        if( line.length() < 11 ) {
            throw new InvalidHexFileException( "Line \""+line+"\" is to short" );
        }
        if( ! ":".equals( line.substring(0,1) ) ) {
            throw new InvalidHexFileException( "Line \""+line
                                               +"\" does not start with ':'" );
        }
        HexFileLine hexLine = new HexFileLine();
        // -- record length --
        try {
            hexLine.setRecordLength( Integer.parseInt( line.substring(1,3), 16 ) );
        } catch ( NumberFormatException ex ) {
            throw new InvalidHexFileException( "Invalid record length on line \""
                                               +line+"\"" );
        }
        // -- address --
        try {
            hexLine.setAddress(Integer.parseInt( line.substring(3,7), 16 ));
        } catch ( NumberFormatException ex ) {
            throw new InvalidHexFileException( "Invalid address on line \""
                                               +line+"\"" );
        }
        // -- record type --
        try {
            hexLine.setRecordType(Integer.parseInt( line.substring(7,9), 16 ));
        } catch ( NumberFormatException ex ) {
            throw new InvalidHexFileException( "Invalid record type on line \""
                                               +line+"\"" );
        }
        if( hexLine.getRecordType() != HexFileLine.DATA_RECORD 
            && hexLine.getRecordType() != HexFileLine.EOF_RECORD ) {
            throw new InvalidHexFileException( "unsupported record type: "
                                               +hexLine.getRecordType() );
        }
        // -- data --        
        try {
            for (int i = 0; i < hexLine.getRecordLength(); i++) {
                int val = Integer.parseInt( line.substring(9+2*i,9+2*i+2), 16 );
                hexLine.setData(i, (byte) val);
            }
        } catch( IndexOutOfBoundsException ex ) {
            throw new InvalidHexFileException(  "data missing on line \""
                    +line+"\"" );            
        } catch( NumberFormatException ex ) {
            throw new InvalidHexFileException( "Invalid data on line \""
                    +line+"\"" );
        }
        // -- checksum --
        try {
            hexLine.setCheckSum(Integer.parseInt( 
                                line.substring(9+2*hexLine.getRecordLength(),
                                               9+2*hexLine.getRecordLength()+2), 
                                               16 ));
        } catch( IndexOutOfBoundsException ex ) {
            throw new InvalidHexFileException(  "checksum missing on line \""
                    +line+"\"" );            
        } catch ( NumberFormatException ex ) {
            throw new InvalidHexFileException( "invalid checksum on line \""
                                               +line+"\"" );
        }
        if( hexLine.getCheckSum() != hexLine.calcCheckSum() ) {
            throw new InvalidHexFileException( "invalid checksum on line \""
                    +line+"\" expected: 0x"+Integer.toHexString(hexLine.calcCheckSum()) );
        }
        return hexLine;
    }

    /**
     * @param rdr The rdr to set.
     */
    public void setRdr(Reader rdr) {
        this.rdr = rdr;
    }

    /**
     * @return Returns the rdr.
     */
    public Reader getRdr() {
        return this.rdr;
    }
    
}
